package net.herit.svcplatform.pushservice.commons.dto.response;

import static org.junit.jupiter.api.Assertions.assertEquals;

import java.util.stream.Stream;

import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;

import com.fasterxml.jackson.core.JsonProcessingException;

import net.herit.svcplatform.pushservice.base.bases.MockitoBase;
import net.herit.svcplatform.pushservice.commons.dto.model.FailModel;
import net.herit.svcplatform.pushservice.commons.dto.response.ErrorResponse;
import net.herit.svcplatform.pushservice.commons.dto.response.HttpResponseStatus;

class HttpResponseStatusTests extends MockitoBase {

	@MethodSource
	@ParameterizedTest(name = "컨트롤러 도착전 에러 발생시, 에러 응답 문자생성 성공 테스트 - {2}")
	@DisplayName("컨트롤러 도착전 에러 발생시, 에러 응답 문자생성 성공 테스트")
	void getErrorResponseToJsonTest(HttpResponseStatus httpResponseStatus, String expected, String displayName) throws JsonProcessingException {
		String actual = HttpResponseStatus.getErrorResponseToJson(httpResponseStatus);
		assertEquals(expected, actual);
	}
	
	private static Stream<Arguments> getErrorResponseToJsonTest() throws JsonProcessingException {
		String invalidResponseJson = getResponseJson(HttpResponseStatus.INVALID_PARAM);
		String notFoundResponseJson = getResponseJson(HttpResponseStatus.NOT_PROVIDE_SERVICE);
		String badRequestResponseJson = getResponseJson(HttpResponseStatus.DUPLICATED_CONTENTS);
		return Stream.of(
	            Arguments.of(
	            		HttpResponseStatus.INVALID_PARAM, 
	            		invalidResponseJson, 
	            		"INVALID_PARAM"),
	            Arguments.of(
	            		HttpResponseStatus.NOT_PROVIDE_SERVICE, 
	            		notFoundResponseJson, 
	            		"NOT_FOUND"),
	            Arguments.of(
	            		HttpResponseStatus.DUPLICATED_CONTENTS, 
	            		badRequestResponseJson, 
	            		"BAD_REQUEST")
	    );
	}
	
	private static String getResponseJson(HttpResponseStatus httpResponseStatus) throws JsonProcessingException {
		ErrorResponse response = new ErrorResponse();
		FailModel failModel = new FailModel();
		failModel.setCode(httpResponseStatus.getCode());
		failModel.setDescription(httpResponseStatus.getDescription());
		response.setResult(failModel);
		return objectMapper.writeValueAsString(response);
	}
	
	@ValueSource(strings = {
			"{\"result\":{\"code\":0}}"
	})
	@DisplayName("컨트롤러 도착전 응답시, 성공 응답 문자생성 성공 테스트")
	@ParameterizedTest(name = "컨트롤러 도착전 응답시, 성공 응답 문자생성 성공 테스트 - {0}")
	void getSuccessResponseToJsonTest(String expected) throws JsonProcessingException {
		String result = HttpResponseStatus.getSuccessResponseToJson();
		assertEquals(expected, result);
	}
	
	@MethodSource
	@DisplayName("resultCode로 responseCode를 찾는 테스트")
	@ParameterizedTest(name = "resultCode로 responseCode를 찾는 테스트 - {2}")
	void findResCodeByCodeTest(int resCode, int code, String displayName) {
		assertEquals(resCode, HttpResponseStatus.findResCodeByCode(code));
	}

	private static Stream<Arguments> findResCodeByCodeTest() {
	    return Stream.of(
	            Arguments.of(
	            		HttpResponseStatus.SUCCESS.getResCode(), 
	            		HttpResponseStatus.SUCCESS.getCode(),
	            		"SUCCESS"),
	            Arguments.of(
	            		HttpResponseStatus.NOT_PROVIDE_SERVICE.getResCode(), 
	            		HttpResponseStatus.NOT_PROVIDE_SERVICE.getCode(),
	            		"NOT_FOUND"),
	            Arguments.of(
	            		HttpResponseStatus.DUPLICATED_CONTENTS.getResCode(), 
	            		HttpResponseStatus.DUPLICATED_CONTENTS.getCode(), 
	            		"BAD_REQUEST"),
	            Arguments.of(
	            		HttpResponseStatus.SYSTEM_ERROR.getResCode(), 
	            		-23, "INTERNAL_SERVER_ERROR")
	    );
	}

	@MethodSource
	@DisplayName("resultCode로 httpResponseStatus를 찾는 테스트")
	@ParameterizedTest(name = "resultCode로 httpResponseStatus를 찾는 테스트 - {2}")
	void findHttpResponseStatusByCodeTest(HttpResponseStatus httpResponseStatus, int code, String displayName) {
		assertEquals(httpResponseStatus, HttpResponseStatus.findHttpResponseStatusByCode(code));
	}

	private static Stream<Arguments> findHttpResponseStatusByCodeTest() {
	    return Stream.of(
	            Arguments.of(
	            		HttpResponseStatus.SUCCESS, 
	            		HttpResponseStatus.SUCCESS.getCode(),
	            		"SUCCESS"),
	            Arguments.of(
	            		HttpResponseStatus.NOT_PROVIDE_SERVICE, 
	            		HttpResponseStatus.NOT_PROVIDE_SERVICE.getCode(),
	            		"NOT_FOUND"),
	            Arguments.of(
	            		HttpResponseStatus.DUPLICATED_CONTENTS, 
	            		HttpResponseStatus.DUPLICATED_CONTENTS.getCode(),
	            		"BAD_REQUEST"),
	            Arguments.of(
	            		HttpResponseStatus.SYSTEM_ERROR, 
	            		-23, "INTERNAL_SERVER_ERROR")
	    );
	}
}
