package net.herit.svcplatform.pushservice.base.bases;

import static org.springframework.restdocs.headers.HeaderDocumentation.headerWithName;
import static org.springframework.restdocs.headers.HeaderDocumentation.requestHeaders;
import static org.springframework.restdocs.mockmvc.MockMvcRestDocumentation.document;
import static org.springframework.restdocs.mockmvc.MockMvcRestDocumentation.documentationConfiguration;
import static org.springframework.restdocs.operation.preprocess.Preprocessors.preprocessRequest;
import static org.springframework.restdocs.operation.preprocess.Preprocessors.preprocessResponse;
import static org.springframework.restdocs.operation.preprocess.Preprocessors.prettyPrint;
import static org.springframework.restdocs.payload.PayloadDocumentation.fieldWithPath;
import static org.springframework.restdocs.payload.PayloadDocumentation.responseFields;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.jsonPath;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.restdocs.AutoConfigureRestDocs;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.restdocs.RestDocumentationContextProvider;
import org.springframework.restdocs.RestDocumentationExtension;
import org.springframework.restdocs.mockmvc.RestDocumentationResultHandler;
import org.springframework.restdocs.snippet.Snippet;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.ResultActions;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.filter.CharacterEncodingFilter;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.herit.svcplatform.pushservice.base.restdoc.RestDocAttributes;
import net.herit.svcplatform.pushservice.commons.auth.AuthService;
import net.herit.svcplatform.pushservice.commons.dto.model.FailModel;
import net.herit.svcplatform.pushservice.commons.dto.response.ErrorResponse;
import net.herit.svcplatform.pushservice.commons.dto.response.HttpResponseStatus;

@Transactional
@SpringBootTest
@AutoConfigureMockMvc
@AutoConfigureRestDocs
@ExtendWith(RestDocumentationExtension.class)
public class RestDocsBaseWithSpringBoot extends BaseTest {

	protected RestDocumentationResultHandler document;
	
	protected MockMvc mockMvc;

	@MockBean(name = "authService")
	protected AuthService authService;
	
    @Autowired
    protected ObjectMapper objectMapper;

    protected ErrorResponse errResp =ErrorResponse.builder()
			.result(
					new FailModel(
							HttpResponseStatus.API_NOT_DEVELOPED_YET.getCode(), 
							HttpResponseStatus.API_NOT_DEVELOPED_YET.getDescription()
							)
					)
			.build();
    
    protected Snippet defaultRequestHeader;
    
    @BeforeEach
    public void setup(WebApplicationContext context, RestDocumentationContextProvider restDocument) {
    	
    	Mockito.when(authService.authTokenCheck(Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any())).thenReturn(true);
    	
    	this.document = document("{class-name}/{method-name}", preprocessRequest(prettyPrint()), preprocessResponse(prettyPrint()));

    	this.mockMvc = MockMvcBuilders
				.webAppContextSetup(context)
				.alwaysDo(this.document)
				.addFilters(new CharacterEncodingFilter("UTF-8", true))
				.apply(documentationConfiguration(restDocument).snippets().withEncoding("UTF-8"))
				.build();
    	
    	this.errResp = ErrorResponse.builder()
				.result(
						new FailModel(
								HttpResponseStatus.API_NOT_DEVELOPED_YET.getCode(), 
								HttpResponseStatus.API_NOT_DEVELOPED_YET.getDescription()
								)
						)
				.build();
    	
    	
    	this.defaultRequestHeader = requestHeaders(
    			headerWithName("X-SERVICE-ID")
	    			.description("X-SERVICE-ID")	
	    			.attributes(
	    					RestDocAttributes.length("255"), 
	    					RestDocAttributes.format(""),
	    					RestDocAttributes.etc("다수 서비스 앱을 사용하는 경우 서비스를 식별하기 위한 코드. NOTE : 설치기사 앱은 'HEVITON' 으로 고정")
					),
					
				headerWithName("X-APP-TYPE")
					.description("X-APP-TYPE")
					.attributes(
							RestDocAttributes.length("10"), 
							RestDocAttributes.format(""), 
							RestDocAttributes.etc("ios / android")
					),
					
				headerWithName("X-TERMINAL-ID")
					.description("X-TERMINAL-ID")
					.attributes(
							RestDocAttributes.length("50"),
							RestDocAttributes.format(""),
							RestDocAttributes.etc("동일 사용자 계정으로 여러 폰으로 로그인할 경우 별도의 세션 처리 (동시로그인 허용) 앱 설치 후 최초실행 시 UUID 형식의 문자열 형식으로 X-TERMINAL-ID 생성")
					),
					
				headerWithName("X-API-VERSION")
					.description("X-API-VERSION")	
					.attributes(
							RestDocAttributes.length("10"),
							RestDocAttributes.format(""),
							RestDocAttributes.etc("1.0")
					),
					
				headerWithName("X-REFERER")
					.description("X-REFERER")	
					.attributes(
							RestDocAttributes.length("50"),
							RestDocAttributes.format(""),
							RestDocAttributes.etc("페이지 단위를 구분하기 위한 코드. NOTE : 앱로깅 API의 function.code와 동일")
					),
				
				headerWithName("X-TRANSACTION-ID")
					.description("X-TRANSACTION-ID")
					.attributes(
							RestDocAttributes.length("50"),
							RestDocAttributes.format(""),
							RestDocAttributes.etc("서비스 플랫폼 또는 API_GW 에서 생성")
					),
				
				headerWithName("Authorization")
					.description("JWT 인증 토큰")
	    			.attributes(
	    					RestDocAttributes.length(""),
	    					RestDocAttributes.format(""),
	    					RestDocAttributes.etc("JWT 인증 토큰")
					)
    			);
    }
    
    public void apiNotDevelopedYet(String uri) throws Exception {
    	
    	ResultActions result = this.mockMvc.perform(post(uri)
				.contentType(MediaType.APPLICATION_JSON)
				.headers(getTestHeader()));
		result
		.andDo(print())
		.andExpect(status().isBadRequest())
		.andExpect(jsonPath("result.code").value(HttpResponseStatus.API_NOT_DEVELOPED_YET.getCode()))
		.andDo(this.document.document(
				this.defaultRequestHeader,
				responseFields(
						fieldWithPath("result.code")
						.description("API가 기획단계에 있습니다.")
						.attributes(RestDocAttributes.length("Integer.MAX_VALUE")),
						fieldWithPath("result.description")
						.description("API가 기획단계에 있습니다.")
						.attributes(RestDocAttributes.length("100"))
				)));
    }
    
    
    public HttpHeaders getTestHeader() {
		HttpHeaders httpHeaders = new HttpHeaders();
		httpHeaders.add("X-SERVICE-ID", "HEVITON");
		httpHeaders.add("X-APP-TYPE", "android");
		httpHeaders.add("X-TERMINAL-ID", "55a0fa94-5ab8-4498-9517-46876b96db0d");
		httpHeaders.add("X-API-VERSION", "1.0.0");
		httpHeaders.add("X-REFERER", "REFERER");
		httpHeaders.add("X-TRANSACTION-ID", "X-TRANSACTION-ID.20200629115826");
		httpHeaders.add("Authorization", "bearer eyJhbGciOiJIUzI1NiJ9.eyJpc3MiOiJoZXJpdCIsInN1YiI6ImFjZXRrbiIsImF1ZCI6InRlc3QiLCJ0eXAiOiIyIiwiaWF0IjoxNTkzMTMwODAyLCJleHAiOjE1OTQ5NTgxODl9.4xBUAbzdTlRfBhPtM-3Dl3gdLHFDhVuVh_qf6jqtsTY");
		return httpHeaders;
	}
    
    public ResultActions performPostMockMvc(String uri, Object request) throws JsonProcessingException, Exception {
    	
    	ResultActions result = this.mockMvc.perform(post(uri)
				.contentType(MediaType.APPLICATION_JSON)
				.content(objectMapper.writeValueAsString(request))
				.headers(getTestHeader()));
    	
    	return result;
    }
}
