package net.herit.svcplatform.pushservice.features.send;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import feign.AsyncFeign;
import feign.jackson.JacksonDecoder;
import feign.jackson.JacksonEncoder;
import net.herit.common.logger.context.HeritLoggerContext;
import net.herit.svcplatform.pushservice.commons.dto.enums.ApiClasses;
import net.herit.svcplatform.pushservice.commons.dto.enums.AppInfos;
import net.herit.svcplatform.pushservice.commons.dto.enums.SvcClasses;
import net.herit.svcplatform.pushservice.commons.dto.response.HttpResponseStatus;
import net.herit.svcplatform.pushservice.commons.logger.CallLogger;
import net.herit.svcplatform.pushservice.commons.logger.OmsLogger;
import net.herit.svcplatform.pushservice.commons.logger.dto.CallObject;
import net.herit.svcplatform.pushservice.commons.logger.dto.OmsObject;
import net.herit.svcplatform.pushservice.features.feign.PushFeignClient;
import net.herit.svcplatform.pushservice.features.feign.dto.model.PushFeignBody;
import net.herit.svcplatform.pushservice.features.feign.dto.model.PushFeignBodyAccount;
import net.herit.svcplatform.pushservice.features.feign.dto.model.PushFeignBodyMessage;
import net.herit.svcplatform.pushservice.features.kafka.dto.model.PushKafkaMessage;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.net.ConnectException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletionException;

@Service
public class PushRequestService {
    @Autowired
    private CallLogger call;

    @Autowired
    private OmsLogger oms;

    @Autowired
    private ObjectMapper mapper;


    @Value("${feign.push.host}")
    private String pushHost;

    @Value("${feign.push.uri}")
    private String pushUri;

    public void request(String message) {
        CallObject callObject = new CallObject();
        OmsObject omsObject = new OmsObject();

        PushKafkaMessage msg;
        try {
            msg = mapper.readValue(message, PushKafkaMessage.class);

            callObject.init();
            String txId = msg.getHeader().getTxId();
            callObject.setTransactionId(txId);
            HeritLoggerContext.applyLogObject(callObject, omsObject, true);

            omsObject.setReqTime();
            omsObject.setTxId(txId);
            omsObject.setLogType(AppInfos.LOG_TYPE);
            omsObject.setSvcId(AppInfos.SVC_ID);
            omsObject.setAppType(AppInfos.APP_TYPE);
            omsObject.setApiClass(ApiClasses.PUSH_CONSUME.getCode());
            omsObject.addSvcClass(SvcClasses.PUSH_REQ.getCode());

            // push api 전송 규격 생성
            PushFeignBodyMessage feignBody = PushFeignBodyMessage.builder().data(msg.getData().getData())
                    .android(defaultAndroidMap())
                    .title(msg.getData().getTitle())
                    .body(msg.getData().getBody())
                    .ios(defaultIosMap())
                    .build();

            PushFeignBodyAccount feignAccount = PushFeignBodyAccount.builder()
                    .pushToken(msg.getData().getPushToken())
                    .build();

            PushFeignBody reqBody = PushFeignBody.builder()
                    .message(feignBody)
                    .account(feignAccount)
                    .build();

            call.debug("push-reqBody:{}", mapper.writeValueAsString(reqBody));
            omsObject.setR1(msg.getData().getPushToken());

            AsyncFeign.asyncBuilder()
                    .encoder(new JacksonEncoder(new ObjectMapper()))
                    .decoder(new JacksonDecoder(new ObjectMapper()))
                    .target(PushFeignClient.class, pushHost)
                    .send(pushUri, msg.getData().getXProjectId(), reqBody);;

            omsObject.setStatusCode(HttpResponseStatus.SUCCESS.getResCode());
            omsObject.setResultCode(HttpResponseStatus.SUCCESS.getCode());
        } catch (JsonProcessingException e) {
            call.error(e.toString());
            omsObject.setStatusCode(HttpResponseStatus.JSON_PARSING_ERROR.getResCode());
            omsObject.setResultCode(HttpResponseStatus.JSON_PARSING_ERROR.getCode());
            throw new RuntimeException(e);
        } catch (CompletionException e) {
            call.error(e.toString());
            omsObject.setStatusCode(HttpResponseStatus.ERROR_EXT_CONNECT.getResCode());
            omsObject.setResultCode(HttpResponseStatus.ERROR_EXT_CONNECT.getCode());
        } catch (Exception e) {
            call.error(e.toString());
            omsObject.setStatusCode(HttpResponseStatus.SYSTEM_ERROR.getResCode());
            omsObject.setResultCode(HttpResponseStatus.SYSTEM_ERROR.getCode());
        } finally {
            omsObject.setResTime();
            oms.write(omsObject);
        }
    }

    public Map<String, Object> defaultAndroidMap(){
        Map<String, String> notification = new HashMap<>();
        notification.put("click_action", "ALERT");

        Map<String, Object> android = new HashMap<>();
        android.put("notification", notification);

        return android;
    }

    public Map<String, Object> defaultIosMap(){
        Map<String, String> payload = new HashMap<>();
        payload.put("category", "ALERT");

        Map<String, Object> ios = new HashMap<>();
        ios.put("payload", payload);

        return ios;
    }

    Map<String, String> makeEchallApiRequestHeaders(String xProjectId) {
        Map<String, String> headers = new HashMap<>();
        headers.put("Content-Type", "application/json");
        headers.put("X-PROJECT-ID", xProjectId);

        return headers;
    }

}
