package net.herit.svcplatform.pushservice.commons.dto.response;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.herit.svcplatform.pushservice.commons.dto.model.FailModel;

public enum HttpResponseStatus {

	/*
	 * 알수없는 예외일 경우 쉽게 검색하기 위해 400 등의 응답 코드와 똑같은 코드로 정의하고, 더 세부적인 resultCode를 중복없이 만들기!
	 * responseCode는 중복이 많아 검색이 힘들기때문.
	 */
	SUCCESS(200, 0, ""),
	NOT_PROVIDE_SERVICE(404, 1, "not support : "),
	INVALID_PARAM(400, 2, "invalid parameter"),
	DUPLICATED_CONTENTS(400, 3, "duplicated contents"),
	SYSTEM_ERROR(500, 4, "Exception"),
	NOT_FOUND_METHOD(405, 5, "Request method not support"),
	
	INVALID_OAUTH_TOKEN(401, 101, "invalid oauth token"),
	INVALID_AUTH_TOKEN(401, 102, "invalid auth token"),
	INVALID_JWT_TOKEN(401, 103, "invalid json web token"),
	NOT_EXIST_USER(401, 104, "not exist user"),
	WRONG_PASSWORD(401, 105, "wrong password"),
	WRONG_GROUP_ID(401, 106, "wrong group id"),
	ALREADY_EXIST_USER(401, 107, "already exist user"),

	JSON_PARSING_ERROR(500, 108, "json parsing error"),
	ERROR_EXT_CONNECT(500, 109, "error ext connection"),

	API_NOT_DEVELOPED_YET(400, 1000, "API 기획중으로 아직 개발되지 않았습니다."),
	API_NOT_PROVIDED_YET(400, 1001, "해당 API는 아직 지원하지 않습니다."),
	;
	
	int resCode;
	int code;
	String reason;
	
	/*
	 * code로 responseStatus를 찾을 일은 없겠지만, 혹시 몰라 선언.
	 * 초기 static 생성자에서 Map을 설정해주는 비용이 들지만, 
	 * HashMap을 이용하기 때문에 이후 code로 뭔가를 찾을때는 반복문을 돌리는것보다 수 배 이상 빠르다고함.
	 * reason은 문자열이라서 검색하기 어렵고, resCode는 중복이 너무 많아 키 값으로 부적절하기 때문에 생략...
	 */
	private static final Map<Integer, HttpResponseStatus> findHttpResponseStatusByCode;
	static {
		findHttpResponseStatusByCode = new ConcurrentHashMap<>();
		for (HttpResponseStatus status : HttpResponseStatus.values()) {
			findHttpResponseStatusByCode.put(status.getCode(), status);
		}
	}
	
	HttpResponseStatus(int resCode, int code, String description) {
		this.resCode = resCode;
		this.code = code;
		this.reason = description;
	}
	
	public int getCode() {
		return code;
	}
	
	public int getResCode() {
		return resCode;
	}

	public String getDescription() {
		return reason;
	}
	
	/**
	 * 컨트롤러까지 오기 전에는 ResponseEntity 객체를 이용한 응답을 못하므로 문자열로 응답해야 하는 경우에 사용.
	 */
	public static String getErrorResponseToJson(HttpResponseStatus httpResponseStatus) throws JsonProcessingException {
		ErrorResponse responseBody = new ErrorResponse();
		responseBody.setResult(new FailModel(httpResponseStatus.getCode(), httpResponseStatus.getDescription()));
		return getResponseToJson(responseBody);
	}
	
	/**
	 * 컨트롤러까지 오기 전에는 ResponseEntity 객체를 이용한 응답을 못하므로 문자열로 응답해야 하는 경우에 사용.
	 */
	public static String getSuccessResponseToJson() throws JsonProcessingException {
		SuccessResponse responseBody = new SuccessResponse();
		return getResponseToJson(responseBody);
	}
	
	private static String getResponseToJson(Object responseBody) throws JsonProcessingException {
		ObjectMapper objectMapper = new ObjectMapper();
		return objectMapper.writeValueAsString(responseBody);
	}
	
	public static Integer findResCodeByCode(int code) {
		return nullReturnServerError(findHttpResponseStatusByCode.get(code)).getResCode();
	}
	
	public static HttpResponseStatus findHttpResponseStatusByCode(int code) {
		return nullReturnServerError(findHttpResponseStatusByCode.get(code));
	}
	
	private static HttpResponseStatus nullReturnServerError(HttpResponseStatus httpResponseStatus) {
		if(httpResponseStatus == null) {
			httpResponseStatus = HttpResponseStatus.SYSTEM_ERROR;
		}
		return httpResponseStatus;
	}
	
}
