package net.herit.svcplatform.pushservice.commons.validation.body;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Valid;
import javax.validation.Validation;
import javax.validation.ValidationException;
import javax.validation.Validator;
import javax.validation.ValidatorFactory;
import javax.validation.constraints.NotNull;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

import net.herit.svcplatform.pushservice.base.bases.MockitoBase;
import net.herit.svcplatform.pushservice.commons.validation.CustomValidation;
import lombok.AllArgsConstructor;
import lombok.Data;

class CustomValidTests extends MockitoBase {

	private static Validator validator;
	
    @BeforeEach
    public void setUp() {        
        ValidatorFactory factory = Validation.buildDefaultValidatorFactory();
        validator = factory.getValidator();
    }
    
    @Test
    @DisplayName("성공 테스트")
    void validationSuccessTest() {
    	ValidationTestObject validationTestObject = getTestObejct("Kim", 22, 30);
    	Set<ConstraintViolation<ValidationTestObject>> constraintViolations = validator.validate(validationTestObject);
    	assertNotNull(constraintViolations);
    	assertTrue(constraintViolations.isEmpty());
    }
    
    @Test
    @DisplayName("@NotNull 실패 테스트")
    void validationNullFailTest() {
    	ValidationTestObject validationTestObject = getTestObejct("Kim", 2, null);
		Set<ConstraintViolation<ValidationTestObject>> constraintViolations = validator.validate(validationTestObject);
    	assertFalse(constraintViolations.isEmpty());
     }

    @Test
    @DisplayName("valid()에서 나이확인 실패 테스트")
    void validationAgeFailTest() {
    	ValidationTestObject validationTestObject = getTestObejct("Kim", 2, 30);
		assertThrows(ValidationException.class, () -> validator.validate(validationTestObject));
    }

    @Test
    @DisplayName("valid()에서 이름확인 실패 테스트")
    void validationNameFailTest() {
    	ValidationTestObject validationTestObject = getTestObejct("KIM", 22, 30);
		assertThrows(ValidationException.class, () -> validator.validate(validationTestObject));
    }
    
    @Test
    @DisplayName("valid()에서 가장 좋아하는 숫자 실패 테스트")
    void validationFavoriteNumberFailTest() {
    	ValidationTestObject validationTestObject = getTestObejct("Kim", 22, 3);
		assertThrows(ValidationException.class, () -> validator.validate(validationTestObject));
    }

    @Test
    @DisplayName("isValid() 메서드의 성공 로직 테스트 (validation 기능 검사아님)")
    void validSuccessResultTest() {
    	ValidationTestObject validationTestObject = getTestObejct("Kim", 22, 30);
    	String isValidResult = validationTestObject.isValid();
    	assertEquals("SUCCESS", isValidResult);
    }
    
    @Test
    @DisplayName("isValid() 메서드의 실패 로직 테스트 (validation 기능 검사아님)")
    void validAllFailResultTest() {
    	ValidationTestObject validationTestObject = getTestObejct("KIM", 2, 3);
    	String isValidResult = validationTestObject.isValid();
    	assertEquals("favorite number fail, name fail, age fail", isValidResult);
    }

    private ValidationTestObject getTestObejct(String name, int age, int favoriteNumber) {
    	return new ValidationTestObject(
    					name, 
    					age, 
    					new ValidationTestObjectInside(favoriteNumber))
    	;
    }
    
    private ValidationTestObject getTestObejct(String name, int age, ValidationTestObjectInside validationTestObjectInside) {
    	return new ValidationTestObject(
    					name, 
    					age, 
    					validationTestObjectInside)
    	;
    }
    
    @Data
    @AllArgsConstructor
    private class ValidationTestObject implements CustomValidation{
    	
    	@NotNull
    	private String name;
    	private int age;
    	@NotNull @Valid
    	private ValidationTestObjectInside validationTestObjectInside;

		@Override
		public String isValid() {
			String result = "";
			
			if(validationTestObjectInside.getFavoriteNumber() < 10)
				result = "favorite number fail, ";
			if(name.equals("KIM")) {
				result += "name fail, ";
			}
			if(age < 20)
				result += "age fail";
			
			return result.equals("") ? "SUCCESS" : result;
		}
		
    }
    
	@Data
	@AllArgsConstructor
	private class ValidationTestObjectInside {
		@NotNull
		private int favoriteNumber;
	}
    
}
