package net.herit.svcplatform.pushservice.commons.util;

import java.security.Key;
import java.util.Calendar;
import java.util.Date;

import javax.crypto.spec.SecretKeySpec;
import javax.xml.bind.DatatypeConverter;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.JwtBuilder;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import net.herit.svcplatform.pushservice.commons.jwt.properties.JwtProperties;

@Component
public class JwtUtil {
	private final String ACCESS_SECRET_KEY;		// ACCESS_TOKEN secret KEY
	private final String REFRESH_SECRET_KEY;	// REFRESH_TOKEN secret KEY
	private	final String ISSUER;				// 토큰발급자
	private final String ACCESS_TOKEN_SUBJECT; 	// ACCESS_TOKEN 토큰제목
	private final String REFRESH_TOKEN_SUBJECT; // REFRESH_TOKEN 토큰제목	
	private final String AUDIENCE; 				// 토큰대상자
	private final String TYPE_KEY;				// 토큰유형  서버
	private final String TYPE_CLICENT;			// 토큰유형1 클라이언트
	private final String TYPE_SERVER;			// 토큰유형2 서버
	
	@Autowired
	public JwtUtil(JwtProperties jwtProperties) {
		ACCESS_SECRET_KEY = jwtProperties.getAccessTokenSecret();
		REFRESH_SECRET_KEY = jwtProperties.getRefreshTokenSecret();
		ISSUER = jwtProperties.getIssuer();
		ACCESS_TOKEN_SUBJECT = jwtProperties.getAccessTokenSecret();
		REFRESH_TOKEN_SUBJECT = jwtProperties.getRefreshTokenSecret();
		AUDIENCE = jwtProperties.getAudience();
		TYPE_KEY = jwtProperties.getTypeKey();
		TYPE_CLICENT = jwtProperties.getTypeClient();
		TYPE_SERVER = jwtProperties.getTypeServer();
	}

	/**
	 * 서버 액세스토큰 발급
	 * @param audience 토큰발급자
	 * @param ttlMillis 만료시간
	 * @return String 토큰
	 */
	public String createServerAccessToken(String adu, int hour) {
		Date now = new Date(System.currentTimeMillis());
		Date exp = getCalculationTokenExpirationDate(hour);
		
		SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS256;
		byte[] apiKeySecretBytes = DatatypeConverter.parseBase64Binary(ACCESS_SECRET_KEY);
		Key signingKey = new SecretKeySpec(apiKeySecretBytes, signatureAlgorithm.getJcaName());
		
		JwtBuilder builder = Jwts.builder().setIssuer(ISSUER)
				.setSubject(ACCESS_TOKEN_SUBJECT)
				.setAudience((adu!=null || !"".equals(adu))? adu:AUDIENCE)
				.claim(TYPE_KEY, TYPE_SERVER)
				.setIssuedAt(now)
				.setExpiration(exp)
				.signWith(signatureAlgorithm, signingKey);
		
		return builder.compact();
	}
	
	/**
	 * 클라이언트 액세스토큰 발급
	 * @param audience 토큰발급자
	 * @param ttlMillis 만료시간
	 * @return String 토큰
	 */
	public String createClientAccessToken(String adu, int hour) {
		Date now = new Date(System.currentTimeMillis());
		Date exp = getCalculationTokenExpirationDate(hour);		
				
		SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS256;
		byte[] apiKeySecretBytes = DatatypeConverter.parseBase64Binary(ACCESS_SECRET_KEY);
		Key signingKey = new SecretKeySpec(apiKeySecretBytes, signatureAlgorithm.getJcaName());
		
		JwtBuilder builder = Jwts.builder().setIssuer(ISSUER)
				.setSubject(ACCESS_TOKEN_SUBJECT)
				.setAudience((adu!=null || !"".equals(adu))? adu:AUDIENCE)
				.claim(TYPE_KEY, TYPE_CLICENT)
				.setIssuedAt(now)
				.setExpiration(exp)
				.signWith(signatureAlgorithm, signingKey);
		
		return builder.compact();
	}
	
	/**
	 * 액세스토큰 해석
	 * @param jwt 토큰데이터
	 * @return Claims 해석데이터 Map형식과 유사
	 */
	public Claims decodeAccessToken(String jwt) {
		return Jwts.parser()
				.setSigningKey(DatatypeConverter.parseBase64Binary(ACCESS_SECRET_KEY))
				.parseClaimsJws(jwt).getBody();
	}
	
	/**
	 * 리프레시토큰 발급
	 * @param adu
	 * @param ttlMillis
	 * @return String 토큰
	 */
	public String createRefreshToken(String adu, int hour) {
		Date now = new Date(System.currentTimeMillis());
		Date exp = getCalculationTokenExpirationDate(hour);	
		
		SignatureAlgorithm signatureAlgorithm = SignatureAlgorithm.HS256;
		byte[] apiKeySecretBytes = DatatypeConverter.parseBase64Binary(REFRESH_SECRET_KEY);
		Key signingKey = new SecretKeySpec(apiKeySecretBytes, signatureAlgorithm.getJcaName());
		
		JwtBuilder builder = Jwts.builder().setIssuer(ISSUER)
				.setSubject(REFRESH_TOKEN_SUBJECT)
				.setAudience((adu!=null || !"".equals(adu))? adu:AUDIENCE)				
				.setIssuedAt(now)
				.setExpiration(exp)
				.signWith(signatureAlgorithm, signingKey);
		
		return builder.compact();
	}
	
	/**
	 * 리프래시 토큰 해석
	 * @param jwt
	 * @return Claims 해석데이터 Map형식과 유사
	 */
	public Claims decodeRefreshToken(String jwt) {
		return Jwts.parser()
				.setSigningKey(DatatypeConverter.parseBase64Binary(REFRESH_SECRET_KEY))
				.parseClaimsJws(jwt).getBody();
	}

	public Date getCalculationTokenExpirationDate(int hour) {
		Date now = new Date(System.currentTimeMillis());
		Calendar cal = Calendar.getInstance();
		cal.setTime(now);		
		cal.add(Calendar.HOUR, hour);		
		return new Date(cal.getTimeInMillis());
	}
	
}
