package net.herit.svcplatform.pushservice.commons.logger.interceptor;

import java.lang.reflect.Field;
import java.sql.Statement;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.ibatis.executor.statement.StatementHandler;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.ParameterMapping;
import org.apache.ibatis.plugin.Interceptor;
import org.apache.ibatis.plugin.Intercepts;
import org.apache.ibatis.plugin.Invocation;
import org.apache.ibatis.plugin.Plugin;
import org.apache.ibatis.plugin.Signature;
import org.apache.ibatis.session.ResultHandler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import net.herit.svcplatform.pushservice.commons.logger.CallLogger;

@Intercepts({
	@Signature(type = StatementHandler.class, method = "update", args = {Statement.class}),
	@Signature(type = StatementHandler.class, method = "query", args = {Statement.class, ResultHandler.class})
})
public class SqlLoggingInterceptor implements Interceptor {
	
	@Autowired
	private CallLogger callLogger;
	
	@Value("${logging.call.multiline}")
	private boolean multiline;
	
	@Value("${logging.call.except-query-log}")
	private boolean exceptQueryLog;
	
	@Override
	public Object intercept(Invocation invocation) throws Exception {
		
		long start = System.currentTimeMillis();
		Object origin = invocation.proceed();
		long end = System.currentTimeMillis();
		
		if(!exceptQueryLog) {
			
			StatementHandler handler = (StatementHandler) invocation.getTarget();
			Object parameter = handler.getParameterHandler().getParameterObject();
			BoundSql boundSql = handler.getBoundSql();
			

			String sql = boundSql.getSql();
			sql = getConstantQuery(sql, parameter, boundSql);
			StringBuilder sb = new StringBuilder();

			if(multiline) {
				sb.append(System.lineSeparator());
				sb.append("Time (millisecond)").append(" >> ").append(end-start).append(System.lineSeparator());
				sb.append("Query").append(" >> ").append(sql).append(System.lineSeparator());
				sb.append("Param").append(" >> ").append(parameter);
			} else {
				sql = sql.replaceAll("\n", "").replaceAll("\t", " ");
				sb.append("Time (millisecond)").append(" >> ").append(end-start).append(", ");
				sb.append("Query").append(" >> ").append(sql).append(", ");
				sb.append("Param").append(" >> ").append(parameter);
			}
			
			callLogger.info("{}", sb.toString());
			
		}
		
		return origin;
	}

	@Override
	public Object plugin(Object target) {
		return Plugin.wrap(target, this);
	}

	@Override
	public void setProperties(Properties properties) {
		Interceptor.super.setProperties(properties);
	}

	public String getConstantQuery(String sql, Object parameter, BoundSql boundSql ) throws IllegalArgumentException, IllegalAccessException, NoSuchFieldException, SecurityException {
		
		if (parameter == null) {
			
			sql = sql.replaceFirst("\\?", "''");
			
		} else {
			
			if(parameter instanceof Integer || parameter instanceof Long || parameter instanceof Float || parameter instanceof Double) {
				
				sql = sql.replaceFirst("\\?", parameter.toString());
				
			} else if(parameter instanceof String) {
				
				sql = sql.replaceFirst("\\?", "'" + parameter.toString() + "'");
				
			} else if (parameter instanceof Map) {
				
				List<ParameterMapping> paramMapping = boundSql.getParameterMappings();
				
				Object value = null;
				
				for(ParameterMapping mapping : paramMapping) {
					
					String key = mapping.getProperty();
					
					if(boundSql.hasAdditionalParameter(key)) {
						value = boundSql.getAdditionalParameter(key);
					} else {
						value = ((Map<?, ?>)parameter).get(key);
					}
					
					if(value instanceof String) {
						sql = sql.replaceFirst("\\?", "'" + value + "'");
					} else if(value != null) {
						sql = sql.replaceFirst("\\?", value.toString());
					} else {
						sql = sql.replaceFirst("\\?", "''");
					}
				}
				
			} else {
				
				List<ParameterMapping> paramMapping = boundSql.getParameterMappings();
				Class<? extends Object> paramClass = parameter.getClass();
				
				for(ParameterMapping mapping : paramMapping) {
					
					String key = mapping.getProperty();
					Field field = paramClass.getDeclaredField(key);
					
					if(field == null) {
					
						sql = sql.replaceFirst("\\?", "''");
						
					} else {
						
						field.setAccessible(true);
						
						Class<?> javaType = mapping.getJavaType();
						
						if(String.class == javaType) {
							
							String sqlParam = "'" + field.get(parameter) + "'";
							sql = sql.replaceFirst("\\?", sqlParam);
						} else {
							String sqlParam = null;
							if(field.get(parameter) == null) {
								sqlParam = "NULL";
							} else {
								sqlParam = field.get(parameter).toString();
							}
							sql = sql.replaceFirst("\\?", sqlParam);
						}
					}
				}
			}
		}
		
		return sql;
	}
	
	
}
