package net.herit.svcplatform.pushservice.commons.logger.interceptor;

import java.nio.charset.StandardCharsets;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.multipart.support.StandardMultipartHttpServletRequest;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.herit.svcplatform.pushservice.commons.dto.model.AuthrizedUser;
import net.herit.svcplatform.pushservice.commons.dto.request.RequestBody;
import net.herit.svcplatform.pushservice.commons.dto.request.RequestSession;
import net.herit.svcplatform.pushservice.commons.dto.response.HttpResponseStatus;
import net.herit.svcplatform.pushservice.commons.dto.wrapper.RequestWrapper;
import net.herit.svcplatform.pushservice.commons.logger.CallLogger;
import net.herit.svcplatform.pushservice.commons.logger.LogTracer;
import net.herit.svcplatform.pushservice.commons.logger.OmsLogger;
import net.herit.svcplatform.pushservice.commons.logger.dto.CallObject;
import net.herit.svcplatform.pushservice.commons.logger.dto.OmsObject;
import net.herit.svcplatform.pushservice.commons.logger.dto.model.LoggerConstant;
import net.herit.svcplatform.pushservice.commons.util.BeanUtil;
import net.herit.svcplatform.pushservice.commons.util.DateUtil;
import net.herit.svcplatform.pushservice.commons.util.HttpTypeUtil;
import lombok.RequiredArgsConstructor;
import net.herit.common.logger.context.HeritLoggerContext;
import net.herit.common.logger.properties.LogProperties;

@Component
@RequiredArgsConstructor
public class LoggerInterceptor implements HandlerInterceptor {
	private final DateUtil dateUtil;
	private final OmsLogger omsLogger;
	private final CallLogger callLogger;
	private final HttpTypeUtil httpTypeUtil;
	private final ObjectMapper objectMapper;
	
	@Autowired
	private RequestSession requestSession;

	@Autowired
	private CallObject callObj;
	@Autowired
	private OmsObject omsObj;

	@Value("${logging.call.multiline}")
	private boolean multiline;

	@Override
	public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
			throws Exception {
		setRequestLogObj(request, handler);

		setRequestSession(request);
		setResponse(request, response);

		startLog(request);

		return true;
	}

	private void setRequestLogObj(HttpServletRequest request, Object handler) {
		HeritLoggerContext.applyLogObject(callObj, omsObj); // STEP 1 에서 생성한 로그 객체를 등록하는 과정
		omsObj.setReqTime(dateUtil.generateTimeStamp("yyyyMMddHHmmssSSS"));

		callObj.request(request);
		omsObj.request(request);

		setRequestOms(handler);
	}

	private void setRequestOms(Object handler) {

		if (handler instanceof HandlerMethod) {
			HandlerMethod handlerMethod = (HandlerMethod) handler;
			LogTracer logTracer = handlerMethod.getMethodAnnotation(LogTracer.class);
			omsObj.logTrace(logTracer);
		}
	}

	private void setRequestSession(HttpServletRequest request) {
		requestSession.setFromServletRequest(request);
		requestSession.setUri(request.getRequestURI());
	}

	private void setResponse(HttpServletRequest request, HttpServletResponse response) {
		String transactionId = request.getHeader("X-TRANSACTION-ID");
		response.setHeader("X-TRANSACTION-ID", transactionId);
		setResponseOms();
	}

	private void setResponseOms() {
		setUserId();

		callObj.setMsgType(LogProperties.MSG_TYPE_RES);
		callObj.setDirection(LoggerConstant.SVC_TO_APP);
	}

	private void setUserId() {
		if (requestSession.getBody() != null && requestSession.getBody().getAccount() != null) {

			String userId = requestSession.getBody().getAccount().getLoginId();

			callObj.setUserId(userId);
			omsObj.setUserId(userId);
		}
	}

	private void startLog(HttpServletRequest request) {
		String contentType = request.getContentType();
		String methodType = request.getMethod();

		if (httpTypeUtil.isJsonPost(contentType, methodType) || httpTypeUtil.isJsonGet(contentType, methodType)) {

			requestJsonLog(request);
		} else if (httpTypeUtil.isFormPost(contentType, methodType)
				|| httpTypeUtil.isFormGet(contentType, methodType)) {

			requestFormLog(request);
		} else if (httpTypeUtil.isFormDataPost(contentType, methodType)
				|| httpTypeUtil.isFormDataGet(contentType, methodType)) {

			multipartRequestLog(request);
		} else if (contentType == null) {
			requestBodyNullLog();
		}
	}

	private void requestJsonLog(HttpServletRequest request) {
		RequestWrapper requestWrapper = null;

		if (request instanceof RequestWrapper) {
			requestWrapper = (RequestWrapper) request;

			byte[] requestBodyArray = requestWrapper.getRequestBodyByteArray();

			String requestBody = new String(requestBodyArray, StandardCharsets.US_ASCII);

			try {
				JsonNode account = objectMapper.readTree(requestBody).get("account");
				AuthrizedUser user = objectMapper.treeToValue(account, AuthrizedUser.class);

				RequestBody body = new RequestBody();
				body.setAccount(body.new Account(user));

				requestSession.setBody(body);
				
				requestCallLog(requestBody, true);

			} catch (Exception e) {

				try {
					requestCallLog(requestBody, false);
				} catch (JsonProcessingException e1) {
					e1.printStackTrace();
				}

			}
		}
	}

	private void requestCallLog(String requestBody, boolean isJson) throws JsonProcessingException {
		if (isJson) {
			JsonNode jsonNode = objectMapper.readTree(requestBody);
			requestBody = objectMapper.writeValueAsString(jsonNode);
		}

		callLog(requestBody);
	}

	private void requestFormLog(HttpServletRequest request) {
		RequestWrapper requestWrapper = null;

		if (request instanceof RequestWrapper) {
			requestWrapper = (RequestWrapper) request;

			byte[] requestBodyArray = requestWrapper.getRequestBodyByteArray();

			String requestBody = new String(requestBodyArray, StandardCharsets.US_ASCII);

			requestCallLog(requestBody);

		}
	}

	private void multipartRequestLog(HttpServletRequest request) {
		if (request instanceof StandardMultipartHttpServletRequest) {
			StandardMultipartHttpServletRequest multipartRequest = (StandardMultipartHttpServletRequest) request;
			formDataRequestLog(multipartRequest);
		}
	}

	private void formDataRequestLog(StandardMultipartHttpServletRequest request) {
		StringBuilder sb = new StringBuilder();

		request.getMultiFileMap().forEach((key, value) -> {
			sb.append(key).append('=');

			for (int i = 0; i < value.size(); i++) {
				sb.append(value.get(i).getOriginalFilename());
				sb.append(",");
			}
			sb.deleteCharAt(sb.lastIndexOf(",")).append('&');
		});

		request.getParameterMap().forEach((key, value) -> {
			sb.append(key).append('=');
			for (int i = 0; i < value.length; i++) {
				sb.append(value[i]).append(',');
			}
			sb.deleteCharAt(sb.lastIndexOf(",")).append('&');
		});

		sb.deleteCharAt(sb.lastIndexOf("&"));

		callLog(sb.toString());
	}

	private void requestCallLog(String requestBody) {
		try {
			requestCallLog(requestBody, false);
		} catch (JsonProcessingException e) {
			e.printStackTrace();
		}
	}

	private void requestBodyNullLog() {
		callLog(null);
	}

	private void callLog(String requestBody) {
		String uri = requestSession.getUri();
		String requestHeader = null;

		try {
			requestHeader = requestSession.getHeader().getHeaderMap() == null ? ""
					: objectMapper.writeValueAsString(requestSession.getHeader().getHeaderMap());
		} catch (JsonProcessingException e) {
			callLogger.error("request header parsing error");
		}

		StringBuilder sb = new StringBuilder();
		if (multiline) {
			sb.append(System.lineSeparator()).append("Request Uri").append(" : ").append(uri)
					.append(System.lineSeparator()).append("Header").append(" : ").append(requestHeader)
					.append(System.lineSeparator()).append("Body").append(" : ").append(requestBody)
					.append(System.lineSeparator());
		} else {
			sb.append("Request Uri").append(" : ").append(uri).append(" ").append("Header").append(" : ")
					.append(requestHeader).append(" ").append("Body").append(" : ").append(requestBody);
		}

		callLogger.info(sb.toString());
	}

	@Override
	public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
			ModelAndView modelAndView) throws Exception {
		HttpTypeUtil httpTypeUtil = getHttpTypeUtil();
		
		if (httpTypeUtil != null && httpTypeUtil.isOctetStream(response.getContentType())) {
			String handlerMethod = handler.toString();
			String contentDisposition = response.getHeader("Content-Disposition");
			octetStreamLog(handlerMethod, contentDisposition);
		}
	
		HeritLoggerContext.resetAttributes(); // 로그 객체 초기화
	}
	
	private void octetStreamLog(String handlerMethod, String contentDisposition) {
		omsObj.resultCode(HttpResponseStatus.SUCCESS.getCode()).httpStatus(HttpResponseStatus.SUCCESS.getResCode())
		.responseTime(dateUtil.generateTimeStamp("yyyyMMddHHmmssSSS"));
		
		if (multiline) {
			callLogger.info("\nMethod: {}, \n{}", handlerMethod, contentDisposition);
		} else {
			callLogger.info("Method: {}, {}", handlerMethod, contentDisposition);
		}
		
		omsLogger.write();
	}
	
	private HttpTypeUtil getHttpTypeUtil() {
		HttpTypeUtil bean = null;
		Object obj = BeanUtil.getBean("httpTypeUtil");
		
		if (obj instanceof HttpTypeUtil) {
			bean = (HttpTypeUtil) obj;
		}
		
		return bean;
	}
}