package net.herit.svcplatform.pushservice.commons.exception;

import javax.servlet.http.HttpServletResponse;

import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.validation.ObjectError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;

import net.herit.svcplatform.pushservice.commons.dto.response.ErrorResponse;
import net.herit.svcplatform.pushservice.commons.dto.response.HttpResponseStatus;
import lombok.RequiredArgsConstructor;

/**
 * ExceptionAdvice
 */
@ControllerAdvice
@RequiredArgsConstructor
public class ExceptionAdvice {
	
    /**
     * Api 로직 진행중, 예상한 예외들에 대한 예외처리 메서드 
     */
    @ExceptionHandler(CommonException.class)
    public ResponseEntity<ErrorResponse> commonExceptionHandler(CommonException ex, WebRequest request) {

        ErrorResponse responseBody = ex.getErrorResponse();
		
        return ResponseEntity.status(ex.getResCode()).body(responseBody);
    }

    /**
     * Validation 예외처리 메서드
     */
    @ExceptionHandler({MethodArgumentNotValidException.class})
    public ResponseEntity<ErrorResponse> validationExceptionHandler(MethodArgumentNotValidException ex, WebRequest request) {

		StringBuilder sb = new StringBuilder();

		for (ObjectError error : ex.getBindingResult().getAllErrors()) {
			sb.append(error.getDefaultMessage());
		}

		ErrorResponse responseBody = new CommonException(HttpResponseStatus.INVALID_PARAM.getCode(), HttpResponseStatus.INVALID_PARAM.getResCode(), sb.toString())
				.getErrorResponse();

    	return ResponseEntity.badRequest().body(responseBody);
    }

    /**
     * Method not Supported
     * @param ex
     * @param response
     * @return
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public ResponseEntity<ErrorResponse> requestMethodNotSupportExceptionHandler(
    		HttpRequestMethodNotSupportedException ex,
    		HttpServletResponse response){

    	ErrorResponse responseBody = new CommonException(
    			HttpResponseStatus.NOT_FOUND_METHOD.getCode(), 
    			HttpResponseStatus.NOT_FOUND_METHOD.getResCode(), 
    			HttpResponseStatus.NOT_FOUND_METHOD.getDescription()).getErrorResponse();

    	return ResponseEntity.status(HttpResponseStatus.NOT_FOUND_METHOD.getResCode()).body(responseBody);
    }
    
    /**
     * 요청 Body Missing
     */
    @ExceptionHandler({HttpMessageNotReadableException.class})
    public ResponseEntity<ErrorResponse> requestBodyMissingExceptionHandler(HttpMessageNotReadableException ex, WebRequest request) {

		String description = ex.getMessage();

		ErrorResponse responseBody = new CommonException(HttpResponseStatus.INVALID_PARAM.getCode(), HttpResponseStatus.INVALID_PARAM.getResCode(), description)
				.getErrorResponse();

		return ResponseEntity.badRequest().body(responseBody);
    }
    
	@ExceptionHandler(AopException.class)
	public ResponseEntity<ErrorResponse> aopExceptionHandler(AopException ex, WebRequest request) {

		ErrorResponse responseBody;
		int resCode;

		try {
			responseBody = ex.getErrorResponse();
			resCode = ex.getHttpResponseStatus().getResCode();
			
			return ResponseEntity.status(resCode).body(responseBody);

		} catch (Exception e) {
			e.printStackTrace();
			responseBody = new CommonException(HttpResponseStatus.SYSTEM_ERROR).getErrorResponse();
			HttpStatus httpStatus = HttpStatus.INTERNAL_SERVER_ERROR;

			return ResponseEntity.status(httpStatus).body(responseBody);
		}
	}

    /**
     * 예상하지 못한 예외들에 대한 예외처리 메서드
     * 404 NOT FOUND는 포함되지 않음
     * 404 NOT_FOUND_EXCEPTION은 ${package}.core.exception.ErrorController 확인
     */
    @ExceptionHandler(Exception.class)
    public ResponseEntity<ErrorResponse> exceptionHandler(Exception ex, WebRequest request) {

		ErrorResponse responseBody = new CommonException(
				HttpResponseStatus.SYSTEM_ERROR).getErrorResponse();
		HttpStatus httpStatus = HttpStatus.INTERNAL_SERVER_ERROR;

        return ResponseEntity.status(httpStatus).body(responseBody);
    }
    
}
