"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const babel_parser_1 = __importDefault(require("../babel-parser"));
const resolveExportedComponent_1 = __importDefault(require("./resolveExportedComponent"));
describe('resolveExportedComponent', () => {
    describe('JavaScript', () => {
        it('should return exported components with vuetify format', () => {
            const ast = (0, babel_parser_1.default)().parse('export default baseMixins.extend().extend({})');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return an export default', () => {
            const ast = (0, babel_parser_1.default)().parse('export default {}');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return an es6 export with its name', () => {
            const ast = (0, babel_parser_1.default)().parse('export const test = {}');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.get('test')).not.toBeUndefined();
        });
        it('should return an es6 export with its name even with 2 statements', () => {
            const ast = (0, babel_parser_1.default)().parse([
                'const testTwoLines = {};',
                'export { testTwoLines };'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.get('testTwoLines')).not.toBeUndefined();
        });
        it('should return an es5 export', () => {
            const ast = (0, babel_parser_1.default)().parse('module.exports = {};');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.get('default')).not.toBeUndefined();
        });
        it('should accept a conditional es5 export', () => {
            const ast = (0, babel_parser_1.default)().parse('if(module !== undefined){ module.exports = {};}');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.get('default')).not.toBeUndefined();
        });
        it('should return an es5 export direct', () => {
            const ast = (0, babel_parser_1.default)().parse('exports = {};');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return an es5 exports.variable', () => {
            const ast = (0, babel_parser_1.default)().parse('exports.xxx = {};');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.get('xxx')).not.toBeUndefined();
        });
        it('should return indirectly exported components', () => {
            const ast = (0, babel_parser_1.default)().parse([
                'const test = {}',
                'export default test'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return indirectly exported class style components', () => {
            const ast = (0, babel_parser_1.default)().parse([
                '@Component()',
                'class testClass extends Vue{}',
                'export default testClass'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.get('default')).not.toBeUndefined();
        });
        it('should return indirectly exported components es5', () => {
            const ast = (0, babel_parser_1.default)().parse([
                'const test = {}',
                'module.exports = test'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should only export components', () => {
            const ast = (0, babel_parser_1.default)().parse([
                'export const otherItem = {}',
                'export const myEnum = {foo: 1, baz: 2}',
                'export default {}'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(2);
        });
        it('should return default exported class style components', () => {
            const ast = (0, babel_parser_1.default)().parse([
                '@Component()',
                'export default class Bart extends testComponent {}'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return name exported class style components', () => {
            const ast = (0, babel_parser_1.default)().parse([
                '@Component()',
                'export class Compo1 extends testComponent {}',
                '@Component()',
                'export class Compo2 extends testComponent {}'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(2);
        });
    });
    describe('TypeScript', () => {
        it('should return exported typescript extend style components', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse(['export default Vue.extend({})'].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return exported named typescript extend style components', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse(['export const Foo = Vue.extend({})'].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should return exported typescript extend custom VueConstructor', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse('export default (Vue as VueConstructor<Vue & SomeInterface>).extend({})');
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should detect as operators for ', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse([
                'import { defineComponent } from "@vue/composition-api"',
                'const CeHeader = {}',
                '',
                'const CeHeaderTheme = defineComponent({',
                '	extends: CeHeader',
                '})',
                '',
                'export default CeHeaderTheme as typeof CeHeader'
            ].join('\n'));
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
    });
    describe('immediately exported variables', () => {
        it('should return the properer ievs when "export {foo as bar}"', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse('export { baz as default } from "file/path"');
            const [, iev] = (0, resolveExportedComponent_1.default)(ast);
            expect(iev.default.exportName).toBe('baz');
        });
        it('should extract IEV aliased and return them', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse([
                'import { baz as foo } from "file/path"',
                'export default foo'
            ].join('\n'));
            const [, iev] = (0, resolveExportedComponent_1.default)(ast);
            expect(iev).toMatchInlineSnapshot(`
				{
				  "foo": {
				    "exportName": "baz",
				    "filePath": [
				      "file/path",
				    ],
				  },
				}
			`);
        });
        it('should extract IEV and return them', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse([
                'import foo from "file/path/foo"',
                'export default foo'
            ].join('\n'));
            const [comps, iev] = (0, resolveExportedComponent_1.default)(ast);
            expect(iev).toMatchInlineSnapshot(`
				{
				  "foo": {
				    "exportName": "default",
				    "filePath": [
				      "file/path/foo",
				    ],
				  },
				}
			`);
            expect(comps.size).toBe(0);
        });
        it('should extract IEV and return them when compbined with normal comps', () => {
            const ast = (0, babel_parser_1.default)({ plugins: ['typescript'] }).parse([
                'import foo from "file/path/blah"',
                'export const blah = {}',
                'export default foo'
            ].join('\n'));
            const [comps, iev] = (0, resolveExportedComponent_1.default)(ast);
            expect(iev).toMatchInlineSnapshot(`
				{
				  "foo": {
				    "exportName": "default",
				    "filePath": [
				      "file/path/blah",
				    ],
				  },
				}
			`);
            expect(comps.size).toBe(1);
        });
    });
    describe('normal functions', () => {
        it('should resolve a component if the body returns an object', () => {
            const ast = (0, babel_parser_1.default)().parse(`export default function (cmp){
				return {
					props: {
						test:String
					}
				}
			};`);
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should resolve a component if the body returns an object and if it has a name', () => {
            const ast = (0, babel_parser_1.default)().parse(`export default function test(cmp){
				return {
					props:{
						test:String
					}
				}
			};`);
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should resolve a component if the body returns an object es5 version', () => {
            const ast = (0, babel_parser_1.default)().parse(`module.exports = function (cmp){
				return {
					props:{
						test: String
					}
				}
			};`);
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
    });
    describe('arrow functions', () => {
        it('should resolve a component if the body is a pure object', () => {
            const ast = (0, babel_parser_1.default)().parse(`export default (cmp) => ({
				props: {
					test: String
				}
			});`);
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
        it('should resolve a component if the body returns an object', () => {
            const ast = (0, babel_parser_1.default)().parse(`export default (cmp) => {
				return {
					props: {
						test: String
					}
				}
			};`);
            const [components] = (0, resolveExportedComponent_1.default)(ast);
            expect(components.size).toBe(1);
        });
    });
});
